//  $Id: List.hh 1.6.1.2.1.1.1.13 Thu, 08 Apr 1999 15:23:32 -0700 cengiz $
//
//  Copyright (c) 1994 by the University of Southern California
//  All rights reserved.
//
//  Permission to use, copy, modify, and distribute this software and its
//  documentation in source and binary forms for lawful non-commercial
//  purposes and without fee is hereby granted, provided that the above
//  copyright notice appear in all copies and that both the copyright
//  notice and this permission notice appear in supporting documentation,
//  and that any documentation, advertising materials, and other materials
//  related to such distribution and use acknowledge that the software was
//  developed by the University of Southern California, Information
//  Sciences Institute. The name of the USC may not be used to endorse or
//  promote products derived from this software without specific prior
//  written permission.
//
//  THE UNIVERSITY OF SOUTHERN CALIFORNIA DOES NOT MAKE ANY
//  REPRESENTATIONS ABOUT THE SUITABILITY OF THIS SOFTWARE FOR ANY
//  PURPOSE.  THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR
//  IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
//  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE,
//  TITLE, AND NON-INFRINGEMENT.
//
//  IN NO EVENT SHALL USC, OR ANY OTHER CONTRIBUTOR BE LIABLE FOR ANY
//  SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES, WHETHER IN CONTRACT, TORT,
//  OR OTHER FORM OF ACTION, ARISING OUT OF OR IN CONNECTION WITH, THE USE
//  OR PERFORMANCE OF THIS SOFTWARE.
//
//  Questions concerning this software should be directed to 
//  ratoolset@isi.edu.
//
//  Author(s): Cengiz Alaettinoglu <cengiz@ISI.EDU>
//             WeeSan Lee <wlee@isi.edu>
//             Ramesh Govindan

#ifndef LIST_H
#define LIST_H

#include <cstdio>      // For NULL definition

// These list manipulation routines assume that each object that
// might be on a list is derived from a ListNode.

class ListNode {
public:
   ListNode *forw;
   ListNode *back;

public:
   ListNode() {
      forw = back = this;
   }
   ListNode(const ListNode& b) {
      forw = back = this;
   }
   virtual ~ListNode() {
      __unlink__();
   };
   
#ifdef LISTNODES_CAN_MODIFY_LIST
   void unlink() {
      __unlink__();
   }
   void link(ListNode *before, ListNode *after) {
      __link__(before, after);
   }
#endif

   // Remove us from any list we're on
   void __unlink__() {
      forw->back = back;
      back->forw = forw;
      forw = back = this;
   }

   // Insert us between before and after
   void __link__(ListNode *before, ListNode *after) {
      forw = after;
      back = before;
      before->forw = after->back = this;
   }
};

typedef ListNode DListNode;

// doubly linked list implemented as a circular list 
// the listhead is also a listnode, except its data pointer points to NULL
// this assumes no non-listhead listnode's data pointer can point to NULL
template <class T>
class List : public ListNode {
protected:
   int       length;

public:
   List() : ListNode() {
      length = 0;
   }

   List(const List& b) : ListNode() { 
      // copies the list b, including the data pointed to
      // the data pointed to is copied using its copy constructor
      // and the bits about this link list is changed explicitly.
      // make sure that the data can be correctly copied using a cconst.
      // Can be safely used if T contains only one ListNode in it

      length = 0;
      for (ListNode *e = b.forw; e != (const ListNode *) &b; e = e->forw) {
	 append(new T(* (T *) e));
      }
   }

   ~List(void) { 
      clear(); 
   }

   List& operator=(const List& b) {
      // This routine blindly free's memory of old elements if any
      // This routine assumes that the elements in this are not accessible
      // via other means.
      // If this assumption does not hold for your system, do not use the = op.
      clear();

      for (ListNode *e = b.forw; e != (const ListNode *) &b; e = e->forw) {
	 append(new T(* (T *) e));
      }

      return *this;
   }

   bool isEmpty() const {
      return (forw == (const ListNode *) this);
   }
   bool isSingleton() const {
      return !isEmpty() && forw == back;
   }

   int size() const {
#ifdef LISTNODES_CAN_MODIFY_LIST
      int	sz = 0;
      for (ListNode* e = forw; e != (const ListNode*) this; e = e->forw)
	 sz++;
      return sz;
#else
      return length;
#endif
   }

   void remove(ListNode *t) {
      t->__unlink__();
      length--;
   }

   void unlink() { // destroys the list, but keeps the elements
      for (ListNode *e = forw; e != this; e = forw)
	 e->__unlink__();
      length = 0;
   }

   void clear() { // destroys the list, and destroys the elements
      for (ListNode *e = forw; e != this; e = forw) {
	 e->__unlink__();
	 delete e;
      }
      length = 0;
   }

   // Remove and return the head of the list
   T*
   dequeue() {
      ListNode*	node;

      length--;
      if (forw == this && back == this) {
	 return (T*)NULL;
      }
      node = forw;
      node->remove();
      return (T*) node;
   }

   void append(ListNode *t) {
      insertBefore(this, t);
   }

   void prepend(ListNode *t) {
      insertAfter(this, t);
   }

   void insertBefore(ListNode *e, ListNode *t) {
      t->__link__(e->back, e);
      length++;
   }

   void insertAfter(ListNode *e, ListNode *t) {
      t->__link__(e, e->forw);
      length++;
   }

/*
 * Splices two lists together, whose list heads are given by this and l.
 * The resulting head is this, and l is returned as the empty list.
 */
   void splice(List& l) {
      if (!l.isEmpty()) {
	 l.forw->back = back;
	 back->forw = l.forw;
	 l.back->forw = this;
	 back = l.back;
	
	 length += l.length;

	 // destroy l
	 l.forw = l.back = &l;
	 l.length = 0;
      }
   }

   // traversal routines
   T *head() const {
      return next(this);
   }

   T *tail() const {
      return prev(this);
   }

   T *next(const ListNode *n) const {
      return (n->forw == (const ListNode *) this) ? (T *) NULL : (T *) n->forw;
   }

   T *prev(const ListNode *n) const {
      return (n->back == (const ListNode *) this) ? (T *) NULL : (T *) n->back;
   }
};

template <class T>
class DList : public List<T> {
};

// T must have the following methods
// T.operator ==
template <class T>
class EQList : public List<T> {
public:
   bool operator==(const EQList<T>& b) {
      if (length != b.length)
	 return false;

      ListNode *e, *e2;
      for (e = forw; e != this; e = e->forw) {
	 for (e2 = b.forw; e2 != (const ListNode *) &b; e2 = e2->forw)
	    if (* ((T *) e) == * ((T *) e2))
	       break;
	 if (e2 == (const ListNode *) &b)
	    return false;
      }
      return true;
   }
};

// sorted doubly linked list
// T must have the following methods
// T.operator <
// T.operator ==
template <class T>
class SortedList : public List<T> {
public:
   void append(ListNode *t) {
      insertSorted(t);
   }

   void prepend(ListNode *t) {
      insertSorted(t);
   }

   void insertBefore(ListNode *e, ListNode *t) {
      insertSorted(t);
   }

   void insertAfter(ListNode *e, ListNode *t) {
      insertSorted(t);
   }
     
   void insertSorted(ListNode *t) {
      ListNode *e;
      for (e = forw; e != this && (*((T *) e) < *((T*) t)); e = e->forw)
	 ;

      if (e != this)
	 t->__link__(e->back, e);
      else
	 t->__link__(back, this);

      length++;
   }

   void insertSortedNoDups(ListNode *t) {
      ListNode *e;
      for (e = forw; e != this && (*((T *) e) < *((T*) t)); e = e->forw)
	 ;
      
      if (e != this) {
	 if (! (*((T *) e) == *((T*) t))) {
	    t->__link__(e->back, e);
	    length++;
	 }
      } else {
	 t->__link__(back, this);
	 length++;
      }
   }

   // do a sorted merge
   void splice(SortedList& l) {
      ListNode *t;
      ListNode *e = forw;
      while (!l.isEmpty()) {
	 t = l.forw;
	 t->__unlink__();
	 for (; 
	      e != this && (*((T *) e) < *((T *) t));
	      e = e->forw)
	    ;
	 if (e != this)
	    t->__link__(e->back, e);
	 else
	    t->__link__(back, this);
      }

      length += l.length;
   }

   // do a sorted merge
   void splice(List<T>& l) {
      ListNode *t;
      ListNode *e = forw;
      while (!l.isEmpty()) {
	 t = l.forw;
	 t->__unlink__();
	 insertSorted(t);
      }
   }

   T *find(const ListNode *t) const {
      for (ListNode *e = forw; e != this; e = e->forw)
	 if ((*((T *) e) == *((T*) t)))
	    return (T *) e;
	 else if (!((*((T *) e) < *((T*) t))))
	    return NULL;

      return NULL;
   }
};

template <class T>
class ListIterator {
private:
   T       *current;
   List<T> &list;
public:
   ListIterator(List<T> &list) : list(list) {
      current = list.head();
   }
   operator T*() const { 
      return current; 
   }
   const T* operator()(void) const {
      return current;
   }
   const ListIterator &operator++(int) {
      current = list.next(current);
      return *this;
   }
};


// Added by wlee@isi.edu
// An example to show it really works  :-)
// It is disable by default

#ifdef DEBUG_LIST_HH

#include <iostream.h>

class Point : public ListNode {
  private:
    int x, y;
  public:
    Point(int x, int y) : x(x), y(y) {}
    int getX(void) const { return x; }
    int getY(void) const { return y; }
};

typedef List<Point> Points;
typedef ListIterator<Point> PointsIterator;

void main(void)
{
  Points pts;
  pts.append(new Point(1,1));
  pts.append(new Point(5,5));
  pts.append(new Point(40,50));

  cout << "Tranditional way to loop thru. a list:" << endl;
  for (Point *pt = pts.head(); pt; pt = pts.next(pt))
    cout << '(' << pt->getX() << ',' << pt->getY() << ')' << endl;

  cout << endl;
  cout << "Using iterator to loop thru. a list;" << endl;
  PointsIterator itr(pts);
  while (itr) {
    cout << '(' << itr()->getX() << ',' << itr()->getY() << ')' << endl;
    itr++;
  }
}

#endif

#endif /* LIST_H */
