// $Id: Prefask.cc 1.1.1.1 Wed, 15 Oct 1997 11:28:18 -0700 wlee $
// 
//  Copyright (c) 1994 by the University of Southern California
//  and/or the International Business Machines Corporation.
//  All rights reserved.
//
//  Permission to use, copy, modify, and distribute this software and
//  its documentation in source and binary forms for lawful
//  non-commercial purposes and without fee is hereby granted, provided
//  that the above copyright notice appear in all copies and that both
//  the copyright notice and this permission notice appear in supporting
//  documentation, and that any documentation, advertising materials,
//  and other materials related to such distribution and use acknowledge
//  that the software was developed by the University of Southern
//  California, Information Sciences Institute and/or the International
//  Business Machines Corporation.  The name of the USC or IBM may not
//  be used to endorse or promote products derived from this software
//  without specific prior written permission.
//
//  NEITHER THE UNIVERSITY OF SOUTHERN CALIFORNIA NOR INTERNATIONAL
//  BUSINESS MACHINES CORPORATION MAKES ANY REPRESENTATIONS ABOUT
//  THE SUITABILITY OF THIS SOFTWARE FOR ANY PURPOSE.  THIS SOFTWARE IS
//  PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES,
//  INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
//  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, TITLE, AND 
//  NON-INFRINGEMENT.
//
//  IN NO EVENT SHALL USC, IBM, OR ANY OTHER CONTRIBUTOR BE LIABLE FOR ANY
//  SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES, WHETHER IN CONTRACT,
//  TORT, OR OTHER FORM OF ACTION, ARISING OUT OF OR IN CONNECTION WITH,
//  THE USE OR PERFORMANCE OF THIS SOFTWARE.
//
//  Questions concerning this software should be directed to 
//  info-ra@isi.edu.
//
//  Author(s): Cengiz Alaettinoglu (cengiz@isi.edu)

#include <cstdio>
#include <cstring>
#include "Prefask.h"


char* int2quad(char *buffer, unsigned int i) {
   sprintf(buffer, "%d.%d.%d.%d", 
	   (i >> 24) & 0xFF, (i >> 16) & 0xFF, (i >> 8)  & 0xFF, i & 0xFF);
   return buffer;
}

unsigned int quad2int(char *quad) {
   unsigned int i;
   unsigned int i1, i2, i3, i4;

   sscanf(quad, "%u.%u.%u.%u", &i1, &i2, &i3, &i4);
   i = i4 + (i3 << 8) + (i2 << 16) + (i1 << 24);
   return i;
}

unsigned int ones(unsigned char from, unsigned char to)
{
  unsigned int result = 0;
  for (int i = 32 - to; i <= 32 - from; i++) result |= (1L << i);
  return result;
}

unsigned int get_mask(unsigned char length)
{
  int mask;

  if (length == 0)
    mask = 0;
  else 
    mask = ~ (0u) << (32 - length);
  
  return mask;  
}

ostream& operator<<(ostream& stream, Prefask& p) {
   char buffer[64];
   stream << p.get_address(buffer);
   return stream;
}


Prefask::Prefask(void) : 
  prefix(0), 
  length(0), 
  n(0), 
  m(0)
{
}

Prefask::Prefask(unsigned int prefix, unsigned char length, 
		 unsigned char n, unsigned char m) :
  prefix(prefix),
  length(length),
  n(n),
  m(m)
{
}

Prefask::Prefask(char *name) : 
  prefix(0), 
  length(0), 
  n(0), 
  m(0)
{
  parse(name);
}

void Prefask::parse(char *name)
{
  unsigned int i1, i2, i3, i4, uiLength = 0, uiN = 0, uiM = 0;
  unsigned int mask;
  char ch = ' ';

  char *p;
  if (strchr(name, '+'))
    // Inclusive more specific operation
    sscanf(name, "%u.%u.%u.%u/%u^%c", &i1, &i2, &i3, &i4, &uiLength, &ch);
  else
    if (p = strchr(name, '-'))
      {
      if (*(p + 1) == 0)
	// Exclusive more specific operation
	sscanf(name, "%u.%u.%u.%u/%u^%c", &i1, &i2, &i3, &i4, &uiLength, &ch);
      else
	// With length specific
	sscanf(name, "%u.%u.%u.%u/%u^%u-%u", 
	       &i1, &i2, &i3, &i4, &uiLength, &uiN, &uiM);
      }
    else
      // Normal address/prefix
      sscanf(name, "%u.%u.%u.%u/%u", &i1, &i2, &i3, &i4, &uiLength);

  length = uiLength;
  n      = uiN;
  m      = uiM;

  switch (ch)
    {
    case '+':
      // inclusive more specifics operator
      n = length;
      m = 32;
      break;
    case '-':
      // exclusive more specifics operator
      n = length + 1;
      m = 32;
      break;
    default:
      if (n == 0) n = length;
      if (m == 0) m = n;
      break;
    }

  prefix = i4 + (i3 << 8) + (i2 << 16) + (i1 << 24);
  
  if (length == 0)
    mask = 0;
  else 
    mask = ~ (0u) << (32 - length);
  
  prefix &= mask;
}

void Prefask::define(unsigned int prefix, unsigned char length, 
		     unsigned char n, unsigned char m)
{
  this->prefix = prefix;
  this->length = length;
  this->n      = n;
  this->m      = m;
}

void Prefask::print(void)
{
  char buffer[64];
  cout << get_address(buffer);
}

int Prefask::valid(void)
{
  if ((length <= n) && (n <= m)) return 1;
  return 0;
}

Prefask& Prefask::operator=(Prefask& other)
{
  prefix = other.prefix;
  length = other.length;
  n      = other.n;
  m      = other.m;
}

// Does it make sense for the more/less specifics operators???
int Prefask::operator<(Prefask& other) 
{
  return prefix < other.prefix || 
    (prefix == other.prefix && length < other.length);
}

// Does it make sense for the more/less specifics operators???
int Prefask::operator<=(Prefask& other) 
{
  return prefix < other.prefix || 
    (prefix == other.prefix && length <= other.length);
}

int Prefask::operator==(Prefask& other) 
{
  return prefix == other.prefix && length == other.length &&
         n == other.n && m == other.m;
}

int Prefask::compare(Prefask& other) 
{
  if (*this < other)  return -1;
  if (*this == other) return 0;
  return 1;
}

char *Prefask::get_address(char *buffer) 
{
  int2quad(buffer, prefix);
  sprintf(buffer + strlen(buffer), "/%u", length);

  if ((length == n) && (n == m))
    // just one route
    ;
  else
    if ((length == n) && (m == 32))
      // inclusive more specifics operator
      strcat(buffer, "^+");
    else
      if ((length == n - 1) && (m == 32))
	// exclusive more specifics operator
	strcat(buffer, "^-");
      else
	if (n == m)
	  sprintf(buffer + strlen(buffer), "^%u", n);
	else
	  sprintf(buffer + strlen(buffer), "^%u-%u", n, m);
  return buffer;
}

unsigned int Prefask::get_mask() 
{  
  return ::get_mask(length);
}
