// $Id: Node.cc 1.3.1.1 Wed, 26 Nov 1997 14:55:25 -0800 cengiz $
// 
//  Copyright (c) 1994 by the University of Southern California
//  and/or the International Business Machines Corporation.
//  All rights reserved.
//
//  Permission to use, copy, modify, and distribute this software and
//  its documentation in source and binary forms for lawful
//  non-commercial purposes and without fee is hereby granted, provided
//  that the above copyright notice appear in all copies and that both
//  the copyright notice and this permission notice appear in supporting
//  documentation, and that any documentation, advertising materials,
//  and other materials related to such distribution and use acknowledge
//  that the software was developed by the University of Southern
//  California, Information Sciences Institute and/or the International
//  Business Machines Corporation.  The name of the USC or IBM may not
//  be used to endorse or promote products derived from this software
//  without specific prior written permission.
//
//  NEITHER THE UNIVERSITY OF SOUTHERN CALIFORNIA NOR INTERNATIONAL
//  BUSINESS MACHINES CORPORATION MAKES ANY REPRESENTATIONS ABOUT
//  THE SUITABILITY OF THIS SOFTWARE FOR ANY PURPOSE.  THIS SOFTWARE IS
//  PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES,
//  INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
//  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, TITLE, AND 
//  NON-INFRINGEMENT.
//
//  IN NO EVENT SHALL USC, IBM, OR ANY OTHER CONTRIBUTOR BE LIABLE FOR ANY
//  SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES, WHETHER IN CONTRACT,
//  TORT, OR OTHER FORM OF ACTION, ARISING OUT OF OR IN CONNECTION WITH,
//  THE USE OR PERFORMANCE OF THIS SOFTWARE.
//
//  Questions concerning this software should be directed to 
//  info-ra@isi.edu.
//
//  Author(s): Cengiz Alaettinoglu (cengiz@isi.edu)

/*
 *[C] The Regents of the University of Michigan and Merit Network, Inc.1993 
 *All Rights Reserved 
 *  
 *  Permission to use, copy, modify, and distribute this software and its 
 *  documentation for any purpose and without fee is hereby granted, provided 
 *  that the above copyright notice and this permission notice appear in all 
 *  copies of the software and derivative works or modified versions thereof, 
 *  and that both the copyright notice and this permission and disclaimer 
 *  notice appear in supporting documentation. 
 *   
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER 
 *   EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION WARRANTIES OF 
 *   MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE REGENTS OF THE 
 *   UNIVERSITY OF MICHIGAN AND MERIT NETWORK, INC. DO NOT WARRANT THAT THE 
 *   FUNCTIONS CONTAINED IN THE SOFTWARE WILL MEET LICENSEE'S REQUIREMENTS OR 
 *   THAT OPERATION WILL BE UNINTERRUPTED OR ERROR FREE. The Regents of the 
 *   University of Michigan and Merit Network, Inc. shall not be liable for any 
 *   special, indirect, incidental or consequential damages with respect to any 
 *   claim by Licensee or any third party arising from use of the software. 
 */

#include "config.hh"
#include <iostream.h>
#include <cstdio>
#include <cstdlib>

#include "Node.h"
#include "NE.hh"
#include "ASMacroMap.h"    
#include "ASMap.h"         
#include "CommunityMap.hh"  
#include "RegisterMap.hh"  
#include "dbase.hh"
#include "radix.hh"
#include "Route.hh"

#define DONTCARE 2

////////////////////////////// ANDNode //////////////////////////////

ANDNode::~ANDNode() {
   delete left;
   delete right;
}

void ANDNode::InOrderPrint() {
   left->InOrderPrint();
   cout << "AND ";
   right->InOrderPrint();
}

void ANDNode::PostOrderPrint() {
   left->InOrderPrint();
   right->InOrderPrint();
   cout << "AND ";
}

void ANDNode::print(ostream &os) {
   os << left << "AND " << right;
}

NormalExpression *ANDNode::Evaluate(int expand) {
   NormalExpression *ne  = left->Evaluate(expand);

   if (ne->empty())
      return ne;

   NormalExpression *ne2 = right->Evaluate(expand);

   ne->do_and(*ne2);

   delete ne2;

   return ne;
}

int ANDNode::match(Route& r, char dontcare) {
   int result1 = left->match(r, dontcare);
   if (result1 == FALSE)
      return FALSE;

   int result2 = right->match(r, dontcare);
   if (result2 == FALSE)
      return FALSE;

   if (result1 == DONTCARE && result2 == DONTCARE)
      return(DONTCARE);

   return(TRUE);
}

////////////////////////////// ORNode //////////////////////////////

ORNode::~ORNode() {
   delete left;
   delete right;
}

void ORNode::InOrderPrint() {
   left->InOrderPrint();
   cout << "OR ";
   right->InOrderPrint();
}

void ORNode::PostOrderPrint() {
   left->InOrderPrint();
   right->InOrderPrint();
   cout << "OR ";
}

void ORNode::print(ostream &os) {
   os << left << "OR " << right;
}

NormalExpression *ORNode::Evaluate(int expand) {
   NormalExpression *ne  = left->Evaluate(expand);

   if (ne->universal())
      return ne;

   NormalExpression *ne2 = right->Evaluate(expand);

   ne->do_or(*ne2);

   delete ne2;

   return ne;
}

int ORNode::match(Route& r, char dontcare) {
   int result1 = left->match(r, dontcare);
   if (result1 == TRUE)
      return TRUE;

   int result2 = right->match(r, dontcare);
   if (result2 == TRUE)
      return TRUE;

   if (result1 == DONTCARE || result2 == DONTCARE)
      return(DONTCARE);

   return(FALSE);
}

////////////////////////////// NotNode //////////////////////////////

NotNode::~NotNode() {
   delete left;
}

void NotNode::InOrderPrint() {
   cout << "NOT (";
   left->InOrderPrint();
   cout << ") ";
}

void NotNode::PostOrderPrint() {
   cout << "(";
   left->InOrderPrint();
   cout << ") NOT ";
}

void NotNode::print(ostream &os) {
   os << "NOT (" << left << ") ";
}

NormalExpression *NotNode::Evaluate(int expand) {
   NormalExpression *ne = left->Evaluate(expand);
   ne->do_not();
   return ne;
}

int NotNode::match(Route& r, char dontcare) {
   int result = left->match(r, !dontcare);
   
   return(result == DONTCARE ? DONTCARE : !result);
}

////////////////////////////// AssignNode //////////////////////////////

AssignNode::~AssignNode() {
   delete left;
}

void AssignNode::InOrderPrint() {
   /* Go left, print then right */
   cout << "set " << (char*) register_map.name(val) << " ";
   left->InOrderPrint();
   cout << " ";
}

void AssignNode::PostOrderPrint() {
   cout << (char*) register_map.name(val) << " ";
   left->InOrderPrint();
   cout << "set ";
}

void AssignNode::print(ostream &os) {
   os << "set " << (char *)register_map.name(val) << " " << left << " ";
}

NormalExpression *AssignNode::Evaluate(int expand) {
   NormalExpression *ne  = left->Evaluate(expand);

   if (register_map.get_ne(val))
      delete register_map.get_ne(val);

   register_map.set_ne(val, ne);

   return NULL;
}

int AssignNode::match(Route& r, char dontcare) {
   return left->match(r, dontcare);
}

////////////////////////////// ANYNode //////////////////////////////

void ANYNode::InOrderPrint() {
   cout << "ANY ";
}

void ANYNode::print(ostream &os) {
   os << "ANY ";
}

NormalExpression *ANYNode::Evaluate(int expand) {
   NormalExpression *ne = new NormalExpression;

   ne->do_not(); // by creation it is NOT ANY

   return ne;
}

int ANYNode::match(Route& r, char dontcare) {
   return TRUE;
}

////////////////////////////// ASNode //////////////////////////////

void ASNode::InOrderPrint() { 
   cout << (char*) AS_map.key( val ) << " ";
} 

void ASNode::print(ostream &os) {
   os << (char *)AS_map.key(val) << " "; 
}

NormalExpression *ASNode::Evaluate(int expand) {
   NormalExpression *ne = new NormalExpression;
   NormalTerm *nt = new NormalTerm;

   Pix as = val;

   if (expand & EXPAND_AS) {
      _SetOfPix s, tmp;
      s.add(as);
      AS2AdrPrfx(s, tmp);
      nt->prfx_set.insert(tmp);
      if (nt->prfx_set.empty()) {
	 delete nt;
	 return ne;
      }
      nt->make_universal(PRFX);       // this makes it universal
      ne->singleton_flag = PRFX;
   } else {
      nt->as_sym.add(as);   
      nt->make_universal(AS_SYM);       // this makes it universal
      ne->singleton_flag = AS_SYM;
   }

   *ne += nt;
   return ne;
}

int ASNode::match(Route& r, char dontcare) {
   if (r.origin.is_dontcare())
      return dontcare;
   return r.origin.match(val) ? TRUE : FALSE;
}

////////////////////////////// ASMacroNode //////////////////////////////

void ASMacroNode::InOrderPrint() { 
   cout << (char*) ASMacro_map.key(val) << " ";
} 

void ASMacroNode::print(ostream &os) {
   os << (char *)ASMacro_map.key(val) << " ";
}

NormalExpression *ASMacroNode::Evaluate(int expand) {
   NormalExpression *ne = new NormalExpression;
   NormalTerm *nt = new NormalTerm;

   if (expand & EXPAND_AS_MACROS) {
      if (expand & EXPAND_AS) {
	 _SetOfPix tmp;
	 AS2AdrPrfx(ASMacro_map.expand(val), tmp);
	 nt->prfx_set.insert(tmp);
	 if (nt->prfx_set.empty()) {
	    delete nt;
	    return ne;
	 }
	 nt->make_universal(PRFX);       // this makes it universal
	 ne->singleton_flag = PRFX;
      } else {
	 // set assignment makes a copy
	 nt->as_sym = ASMacro_map.expand(val);
	 if (nt->as_sym.empty()) {
	    delete nt;
	    return ne;
	 }
	 nt->make_universal(AS_SYM);       // this makes it universal
	 ne->singleton_flag = AS_SYM;
      }
   } else {
      nt->as_macros.add(val);
      nt->make_universal(AS_MACROS);       // this makes it universal
      ne->singleton_flag = AS_MACROS;
   }

   *ne += nt;
   return ne;
}

int ASMacroNode::match(Route& r, char dontcare) {
   if (r.origin.is_dontcare())
      return dontcare;
   
   _SetOfPix& as_set = ASMacro_map.expand(val);

   for (Pix pi = as_set.first(); pi; as_set.next(pi))
      if (r.origin.match(as_set(pi)))
	 return TRUE;

   return FALSE;
}


////////////////////////////// CommNode //////////////////////////////

void CommNode::InOrderPrint() { 
   cout << (char*) Community_map.key( val ) << " ";
} 

void CommNode::print(ostream &os) {
   os << (char *)Community_map.key(val) << " ";
}

NormalExpression *CommNode::Evaluate(int expand) {
   NormalExpression *ne = new NormalExpression;
   NormalTerm *nt = new NormalTerm;

   if (expand & EXPAND_COMMUNITIES) {
      // set assignment makes a copy
      nt->prfx_set = Community_map.expand(val);
      if (nt->prfx_set.empty()) {
	 delete nt;
	 return ne;
      }
      nt->make_universal(PRFX);
      ne->singleton_flag = PRFX;
   } else {
      nt->prfx_macros.add(val);
      nt->make_universal(PRFX_MACROS);       // this makes it universal
      ne->singleton_flag = PRFX_MACROS;
   }

   *ne += nt;
   return ne;
}

int CommNode::match(Route& r, char dontcare) {
   if (r.ripecommunity.is_dontcare())
      return dontcare;

   return r.ripecommunity.match(val) ? TRUE : FALSE;
}

////////////////////////////// NetListNode //////////////////////////////

void NetListNode::InOrderPrint() {
   Pix pi = nets.first();

   cout << "{" << Prefask_map(nets(pi));
   for (nets.next(pi); pi; nets.next(pi))
      cout << ", " << Prefask_map(nets(pi));
   cout << "} ";
}

void NetListNode::print(ostream &os) {
   Pix pi = nets.first();

   os << "{" << Prefask_map(nets(pi));
   for (nets.next(pi); pi; nets.next(pi))
      os << ", " << Prefask_map(nets(pi));
   os << "} ";
}

NormalExpression *NetListNode::Evaluate(int expand) {
   NormalExpression *ne = new NormalExpression;
   NormalTerm *nt = new NormalTerm;

   nt->prfx_set = nets;
   nt->make_universal(PRFX);       // this makes it universal
   ne->singleton_flag = PRFX;

   *ne += nt;
   return ne;
}

int NetListNode::match_exact_or_more_specific(Route& r, char dontcare) {
   if (r.nlri.is_dontcare())
      return DONTCARE;

   if(cidr_root == NULL) {
       cidr_root = radix_create_node(NULL,0,0,NULL,EMPTY);
       for (Pix pi = nets.first(); pi; nets.next(pi)) {
	  radix_insert(Prefask_map(nets(pi)).get_prefix(),
		       (int) Prefask_map(nets(pi)).get_length(),
		       NULL, cidr_root);
       }
   }
			
   int flag = radix_search(Prefask_map(r.nlri.pix).get_prefix(),
			   (int) Prefask_map(r.nlri.pix).get_length(),
			   cidr_root);

   return flag ? TRUE : FALSE;
}

int NetListNode::match(Route& r, char dontcare) {
   if (r.nlri.is_dontcare())
      return DONTCARE;

   for (Pix pi = nets.first(); pi; nets.next(pi))
      if (nets(pi) == r.nlri.pix)
         return TRUE;

   return FALSE;
}

////////////////////////////// ASPathNode //////////////////////////////

ASPathNode::~ASPathNode() {
   if (re)
      delete re;
}

void ASPathNode::InOrderPrint() { 
   cout << "<" << *re << "> ";
} 

void ASPathNode::print(ostream &os) {
   os << "<" << *re << "> ";
}

NormalExpression *ASPathNode::Evaluate(int expand) {
   NormalExpression *ne = new NormalExpression;
   NormalTerm *nt = new NormalTerm;

   nt->as_path = re;
   nt->make_universal(AS_PATH);       // this makes it universal
   ne->singleton_flag = AS_PATH;

   *ne += nt;
   return ne;
}

int ASPathNode::match(Route& r, char dontcare) {
   if (r.aspath.is_dontcare())
      return dontcare;

   if (!re)
      return FALSE;

   return re->match(r.aspath) ? TRUE : FALSE;
}

////////////////////////////// RegisterNode //////////////////////////////

void RegisterNode::InOrderPrint() { 
   cout << (char*) register_map.name(val) << " ";
} 

void RegisterNode::print(ostream &os) {
   os << (char *)register_map.name(val) << " ";
}

NormalExpression *RegisterNode::Evaluate(int expand) {
   NormalExpression *ne;
   
   if (register_map.get_ne(val))
      ne = new NormalExpression(*register_map.get_ne(val));
   else {
      cerr << "Warning: register " << (char*) register_map.name(val) 
	   << " is not defined." << endl;
      ne = new NormalExpression; // by creation it is NOT ANY
   }

   return ne;
}

int RegisterNode::match(Route& r, char dontcare) {
   return dontcare;
}

////////////////////////////// ActionNode //////////////////////////////

int ActionNode::operator< (ActionNode &b) {
   PrefNode *n1 = (PrefNode *) FindFirst(T_PrefNode);
   PrefNode *n2 = (PrefNode *) b.FindFirst(T_PrefNode);
   if (!n2)
      return 1;
   else
      if (!n1)
	 return 0;
      else
	 return  n1->pref() < n2->pref();
}

////////////////////////////// PrefNode //////////////////////////////

void PrefNode::InOrderPrint() { 
   if (val == -1)
      cout << "pref=MED ";
   else
      cout << "pref=" << val << " ";
} 

void PrefNode::print(ostream &os) {
   if (val == -1)
      os << "pref=MED ";
   else
      os << val << " ";   
}

////////////////////////////// MEDNode //////////////////////////////

void MEDNode::InOrderPrint() { 
   if (val == -1)
      cout << "metric-out=IGP ";
   else
      cout << "metric-out=" << val << " ";
} 

void MEDNode::print(ostream &os) {
   if (val == -1)
      os << "metric-out=IGP ";
   else
      os << "metric-out=" << val << " ";
}

////////////////////////////// DPANode //////////////////////////////

void DPANode::InOrderPrint() { 
   cout << "dpa=" << val << " ";
} 

void DPANode::print(ostream &os) {
   os << "dpa=" << val << " ";
}

////////////////////////////// NoopNode //////////////////////////////

void NoopNode::InOrderPrint() { 
   cout << "(noop) ";
} 

void NoopNode::print(ostream &os) {
}

////////////////////////////// ComposeNode //////////////////////////////

ComposeNode::~ComposeNode() {
   delete left;
   delete right;
}

void ComposeNode::InOrderPrint() {
   left->InOrderPrint();
   cout << ";";
   right->InOrderPrint();
}

void ComposeNode::print(ostream &os) {
   os << left << ";" << right;
}
